/** @file
Platform Id based on SMBIOS structures

Copyright (c) 2016. Disk Cryptography Services for EFI (DCS), Alex Kolotnikov
Copyright (c) 2016. VeraCrypt, Mounir IDRASSI

This program and the accompanying materials
are licensed and made available under the terms and conditions
of the GNU Lesser General Public License, version 3.0 (LGPL-3.0).

The full text of the license may be found at
https://opensource.org/licenses/LGPL-3.0
**/

#include <Uefi.h>
#include <Guid/SmBios.h>
#include <IndustryStandard\SmBios.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/BaseMemoryLib.h>

#include "Library/CommonLib.h"

SMBIOS_TABLE_ENTRY_POINT*     gSmbTable = NULL;
EFI_GUID*                     gSmbSystemUUID = NULL;           // Universal unique ID 
CHAR8*                        gSmbSystemSerial = NULL;         // System serial
CHAR8*                        gSmbSystemSKU = NULL;            // SKU number
CHAR8*                        gSmbSystemManufacture = NULL;    // computer manufacture
CHAR8*                        gSmbSystemModel = NULL;          // computer model
CHAR8*                        gSmbSystemVersion = NULL;        // computer version

CHAR8*                        gSmbBaseBoardSerial = NULL;      // Base board serial
UINT64*                       gSmbProcessorID = NULL;          // Processor ID

CHAR8*                        gSmbBiosVendor = NULL;           // BIOS vendor
CHAR8*                        gSmbBiosVersion = NULL;          // BIOS version
CHAR8*                        gSmbBiosDate = NULL;             // BIOS date



UINTN        gBioIndexAuth = 0;
typedef struct _DCS_AUTH_DATA_MARK {
	UINT32     HeaderCrc;
	UINT32     PlatformCrc;
	UINT32     AuthDataSize;
	UINT32     Reserved;
} DCS_AUTH_DATA_MARK;

CHAR8* SMBIOSGetString(UINTN StringNumber, SMBIOS_STRUCTURE* smbtbl, CHAR8* lastAddr) {
	CHAR8* String;
	UINTN Index;
	String = ((UINT8*)smbtbl) + smbtbl->Length;
	for (Index = 1; Index <= StringNumber; Index++) {
		if (StringNumber == Index) {
			return String;
		}
		//
		// Skip string
		//
		while (*String != 0) {
			String++;
			if (String > lastAddr) {
				return NULL;
			}
		}
		String++;
// 		if (*String == 0) {
// 			return NULL;
// 		}
	}
	return NULL;
}

/**
* Get SMBIOS serial data
*/
EFI_STATUS
SMBIOSGetSerials()
{
	EFI_STATUS                    res;
	SMBIOS_STRUCTURE_POINTER      pSMBIOS;
	CHAR8*                        pos = NULL;
	CHAR8*                        endOfTable;

	// Get SMBIOS tables pointer from System Configure table
	res = EfiGetSystemConfigurationTable(&gEfiSmbiosTableGuid, (VOID**)&gSmbTable);
	if (EFI_ERROR(res)) {
		return res;
	}
	pSMBIOS.Raw = (UINT8 *)(UINTN)(gSmbTable->TableAddress);
	pos = pSMBIOS.Raw;
	endOfTable = pSMBIOS.Raw + gSmbTable->TableLength;
	do {
		SMBIOS_STRUCTURE* smbtbl = (SMBIOS_STRUCTURE*)pos;
		// BIOS information
		if (smbtbl->Type == 0) {
			gSmbBiosVendor = SMBIOSGetString(1, smbtbl, endOfTable);
			gSmbBiosVersion = SMBIOSGetString(2, smbtbl, endOfTable);
			gSmbBiosDate = SMBIOSGetString(3, smbtbl, endOfTable);
		}
		// System info
		if (smbtbl->Type == 1) {
			gSmbSystemUUID = (EFI_GUID*)&pos[8];
			gSmbSystemManufacture = SMBIOSGetString(1, smbtbl, endOfTable);
			gSmbSystemModel = SMBIOSGetString(2, smbtbl, endOfTable);
			gSmbSystemVersion = SMBIOSGetString(3, smbtbl, endOfTable);
			gSmbSystemSerial = SMBIOSGetString(4, smbtbl, endOfTable);
			gSmbSystemSKU = SMBIOSGetString(5, smbtbl, endOfTable);
		}
		// Base board
		if (smbtbl->Type == 2) {
			gSmbBaseBoardSerial = SMBIOSGetString(4, smbtbl, endOfTable);
		}
		// Processor
		if (smbtbl->Type == 4) {
			gSmbProcessorID = (UINT64*)&pos[8];
		}
		pos += smbtbl->Length;
		while (((pos[0] != 0) || (pos[1] != 0)) && (pos < endOfTable)) pos++;
		pos += 2;
	} while (pos < endOfTable);

	return EFI_SUCCESS;
}

EFI_STATUS
PlatformGetID(
	IN  EFI_HANDLE  handle,
	OUT CHAR8       **id,
	OUT UINTN       *idlength
	) 
{
	EFI_STATUS                    res = EFI_SUCCESS;
	UINTN                         idLen = 0;
	CHAR8*                        idBuf = NULL;
	CHAR8*                        handleSerial = NULL;

	UsbGetId(handle, &handleSerial);
	if (gSmbSystemUUID == NULL) SMBIOSGetSerials();
	idLen += (gSmbSystemUUID == NULL) ? 0 : sizeof(*gSmbSystemUUID);
	idLen += (gSmbSystemSerial == NULL) ? 0 : AsciiStrLen((char*)gSmbSystemSerial) + 1;
	idLen += (gSmbSystemSKU == NULL) ? 0 : AsciiStrLen((char*)gSmbSystemSKU) + 1;
	idLen += (gSmbBaseBoardSerial == NULL) ? 0 : AsciiStrLen((char*)gSmbBaseBoardSerial) + 1;
	idLen += (gSmbProcessorID == NULL) ? 0 : sizeof(*gSmbProcessorID);
	idLen += (handleSerial == NULL) ? 0 : AsciiStrLen((char*)handleSerial) + 1;

	idBuf = MEM_ALLOC(idLen);
	if (idBuf == NULL) {
		res = EFI_BUFFER_TOO_SMALL;
		goto error;
	}

	*id = idBuf;
	*idlength = idLen;

	if (gSmbSystemUUID != NULL) {
		CopyMem(idBuf, gSmbSystemUUID, sizeof(*gSmbSystemUUID));
		idBuf += sizeof(*gSmbSystemUUID);
	}

	if (gSmbSystemSerial != NULL) {
		UINTN ssz;
		ssz = AsciiStrLen((char*)gSmbSystemSerial) + 1;
		CopyMem(idBuf, gSmbSystemSerial, ssz);
		idBuf += ssz;
	}

	if (gSmbSystemSKU != NULL) {
		UINTN ssz;
		ssz = AsciiStrLen((char*)gSmbSystemSKU) + 1;
		CopyMem(idBuf, gSmbSystemSKU, ssz);
		idBuf += ssz;
	}

	if (gSmbBaseBoardSerial != NULL) {
		UINTN ssz;
		ssz = AsciiStrLen((char*)gSmbBaseBoardSerial) + 1;
		CopyMem(idBuf, gSmbBaseBoardSerial, ssz);
		idBuf += ssz;
	}

	if (gSmbProcessorID != NULL) {
		CopyMem(idBuf, gSmbProcessorID, sizeof(*gSmbProcessorID));
		idBuf += sizeof(*gSmbProcessorID);
	}

	if (handleSerial != NULL) {
		UINTN ssz;
		ssz = AsciiStrLen((char*)handleSerial) + 1;
		CopyMem(idBuf, handleSerial, ssz);
		idBuf += ssz;
		MEM_FREE(handleSerial);
	}

	return res;

error:
	MEM_FREE(handleSerial);
	MEM_FREE(idBuf);
	return res;
}


EFI_STATUS
PlatformGetIDCRC(
	IN  EFI_HANDLE  handle,
	OUT UINT32      *crc32
	) 
{
	EFI_STATUS                    res;
	UINTN                         crcLen;
	CHAR8*                        crcBuf = NULL;
	res = PlatformGetID(handle, &crcBuf, &crcLen);
	if (EFI_ERROR(res)) {
		return res;
	}
	res = gBS->CalculateCrc32(crcBuf, crcLen, crc32);
	MEM_FREE(crcBuf);
	return res;
}

EFI_STATUS
PlatformGetAuthDataByType(
	OUT UINT8        **data, 
	OUT UINTN        *len,
	OUT EFI_HANDLE   *secRegionHandle,
	IN  BOOLEAN      RemovableMedia)
{
	EFI_STATUS                    res;
	UINT32                        crc;
	CHAR8*                        buf = NULL;
	EFI_BLOCK_IO_PROTOCOL*        bio;
	DCS_AUTH_DATA_MARK*           mark = NULL;
	mark = (DCS_AUTH_DATA_MARK*)MEM_ALLOC(512);
	for (; gBioIndexAuth < gBIOCount; ++gBioIndexAuth) {
		bio = EfiGetBlockIO(gBIOHandles[gBioIndexAuth]);
		if (bio == NULL) 	continue;
		if(bio->Media->RemovableMedia != RemovableMedia) continue;
		res = bio->ReadBlocks(bio, bio->Media->MediaId, 61, 512, mark);
		if (EFI_ERROR(res)) continue;
		
		res = gBS->CalculateCrc32(&mark->PlatformCrc, sizeof(*mark) - 4, &crc);
		if (EFI_ERROR(res)) continue;
		if( crc != mark->HeaderCrc) continue;
		
		res = PlatformGetIDCRC(gBIOHandles[gBioIndexAuth], &crc);
		if (EFI_ERROR(res)) continue;
		if (crc != mark->PlatformCrc) continue;
		
		buf = MEM_ALLOC(mark->AuthDataSize * 1024 * 128);
		if (buf == NULL) continue;
		
		res = bio->ReadBlocks(bio, bio->Media->MediaId, 62, mark->AuthDataSize * 1024 * 128, buf);
		if (EFI_ERROR(res)) {
			MEM_FREE(buf);
			continue;
		}
		*data = buf;
		*len = ((UINTN) mark->AuthDataSize) * 1024 * 128;
		*secRegionHandle = gBIOHandles[gBioIndexAuth];
		return EFI_SUCCESS;
	}
	return EFI_NOT_FOUND;
}

BOOLEAN gBioIndexAuthOnRemovable = TRUE;

EFI_STATUS
PlatformGetAuthData(
	OUT UINT8      **data,
	OUT UINTN      *len,
	OUT EFI_HANDLE *secRegionHandle
	)
{
	EFI_STATUS res;
	res = PlatformGetAuthDataByType(data, len, secRegionHandle, gBioIndexAuthOnRemovable);
	if (EFI_ERROR(res)) {
		if (gBioIndexAuthOnRemovable) {
			gBioIndexAuthOnRemovable = FALSE;
			gBioIndexAuth = 0;
			res = PlatformGetAuthDataByType(data, len, secRegionHandle, FALSE);
		}
	}
	return res;
}

CHAR8 gDCS_platform_crt_der[1341] = {
	0x30, 0x82, 0x05, 0x39, 0x30, 0x82, 0x03, 0x21, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x32,
	0xDC, 0x46, 0x30, 0x87, 0xE5, 0x4F, 0xB1, 0x43, 0x0F, 0x58, 0x9E, 0xC0, 0xDA, 0x58, 0xF8, 0x30,
	0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x23,
	0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x1E, 0x18, 0x00, 0x44, 0x00, 0x43, 0x00,
	0x53, 0x00, 0x5F, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x61, 0x00, 0x74, 0x00, 0x66, 0x00, 0x6F, 0x00,
	0x72, 0x00, 0x6D, 0x30, 0x1E, 0x17, 0x0D, 0x31, 0x36, 0x30, 0x38, 0x30, 0x39, 0x30, 0x38, 0x33,
	0x38, 0x31, 0x31, 0x5A, 0x17, 0x0D, 0x33, 0x31, 0x30, 0x38, 0x30, 0x39, 0x30, 0x38, 0x33, 0x38,
	0x31, 0x30, 0x5A, 0x30, 0x23, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x1E, 0x18,
	0x00, 0x44, 0x00, 0x43, 0x00, 0x53, 0x00, 0x5F, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x61, 0x00, 0x74,
	0x00, 0x66, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x6D, 0x30, 0x82, 0x02, 0x22, 0x30, 0x0D, 0x06, 0x09,
	0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x02, 0x0F, 0x00,
	0x30, 0x82, 0x02, 0x0A, 0x02, 0x82, 0x02, 0x01, 0x00, 0xAF, 0x5B, 0x97, 0x06, 0x70, 0x4F, 0x3B,
	0x2E, 0x50, 0x6A, 0xD1, 0x47, 0xCB, 0x70, 0x20, 0xF4, 0x77, 0x79, 0x06, 0xCA, 0xA9, 0xA2, 0x13,
	0x75, 0xAD, 0x07, 0x66, 0x94, 0xC2, 0xBB, 0xCA, 0x7E, 0xFC, 0x6C, 0x19, 0x16, 0x5D, 0x60, 0x77,
	0x6E, 0xCB, 0xF3, 0x8A, 0xC2, 0xF6, 0x53, 0xC7, 0xC2, 0xB1, 0x87, 0x5F, 0x8E, 0xFA, 0x20, 0xDF,
	0xBA, 0x00, 0xCE, 0xBA, 0xA7, 0xC8, 0x65, 0x7E, 0xFC, 0xA8, 0xF8, 0x50, 0x9E, 0xD7, 0x7D, 0x8E,
	0x4F, 0xB1, 0x1B, 0x60, 0xC0, 0xD2, 0xBC, 0x4A, 0xB4, 0x46, 0xA5, 0x0E, 0x90, 0x38, 0xA5, 0x7B,
	0x58, 0xEE, 0x16, 0xD9, 0xBA, 0x73, 0xAD, 0x69, 0x2A, 0xA4, 0xB4, 0x51, 0x0C, 0x21, 0x8C, 0x3D,
	0x0E, 0x40, 0x44, 0x20, 0x2E, 0xE2, 0xEF, 0x16, 0x25, 0xE8, 0x1C, 0xE8, 0xD2, 0x76, 0x66, 0x8E,
	0xA1, 0xB8, 0x29, 0x28, 0x23, 0xA2, 0x9F, 0xCA, 0xAB, 0x0D, 0x81, 0x4A, 0xE0, 0xF9, 0x87, 0x7B,
	0xD6, 0xDA, 0x2E, 0x10, 0x21, 0xBD, 0x69, 0x9C, 0x86, 0x45, 0xD2, 0xE8, 0xCD, 0xA1, 0xF6, 0xC2,
	0x09, 0x93, 0x68, 0x06, 0xA0, 0x5D, 0xB7, 0x2C, 0xD7, 0x83, 0x0B, 0xCC, 0xFE, 0x91, 0x90, 0x1E,
	0x85, 0x96, 0x72, 0xBC, 0x3E, 0x9C, 0xD4, 0x1C, 0xDF, 0xC4, 0x85, 0xB3, 0xD7, 0x00, 0x43, 0xDD,
	0xA8, 0x7C, 0xD1, 0xDE, 0x89, 0xDB, 0x2A, 0x70, 0x27, 0x6F, 0x46, 0xF9, 0x3A, 0x9E, 0x55, 0x10,
	0x5A, 0x82, 0x42, 0x72, 0x42, 0xEA, 0x83, 0x0F, 0x39, 0x3A, 0x50, 0x67, 0xFE, 0x4F, 0x9D, 0x91,
	0x50, 0x93, 0xB3, 0xC6, 0x12, 0x60, 0xAE, 0x3A, 0x5A, 0xB7, 0xB7, 0x9C, 0x83, 0xA0, 0xD2, 0xFF,
	0xFF, 0x23, 0xC3, 0x95, 0x66, 0x79, 0x20, 0xA0, 0x09, 0x02, 0x74, 0x15, 0x34, 0x2A, 0x0A, 0x6E,
	0x80, 0x36, 0x13, 0xC7, 0x9B, 0x77, 0x81, 0x35, 0x45, 0xDD, 0xEC, 0x11, 0xC3, 0x43, 0xA6, 0x48,
	0xF8, 0xDB, 0xC0, 0x3C, 0x12, 0x86, 0x37, 0x68, 0xF4, 0xEA, 0x70, 0x41, 0x66, 0x6D, 0x56, 0x7C,
	0xFC, 0xE8, 0x61, 0xD7, 0x82, 0x02, 0xC6, 0xFD, 0xA5, 0x74, 0xCE, 0xA6, 0x39, 0xFB, 0xD2, 0x21,
	0x61, 0x15, 0x6B, 0x6E, 0x0B, 0xD6, 0x65, 0xF5, 0x8C, 0x5A, 0x52, 0x5E, 0x16, 0x96, 0x02, 0x09,
	0x81, 0x28, 0x32, 0xBF, 0x2C, 0x1E, 0x0F, 0xAD, 0x1E, 0xE5, 0xAD, 0x3B, 0x19, 0x24, 0xED, 0xC1,
	0xA7, 0x60, 0xC9, 0x2D, 0xE4, 0x15, 0xA7, 0xAF, 0x91, 0x35, 0x07, 0x5A, 0x31, 0x39, 0xB1, 0xA5,
	0x3C, 0xE3, 0x59, 0x9A, 0x85, 0xC8, 0x6F, 0x83, 0x6F, 0xFF, 0x3C, 0x81, 0xC1, 0x8F, 0xF6, 0x2E,
	0x3C, 0x1B, 0xF5, 0x9A, 0x21, 0x5D, 0xAD, 0x3A, 0x9B, 0x7F, 0x18, 0x4F, 0x62, 0x09, 0xEA, 0x2F,
	0x5D, 0x15, 0xFD, 0x9D, 0x73, 0x78, 0x95, 0x76, 0x47, 0x15, 0x1C, 0x9A, 0x3F, 0x19, 0xB7, 0xCE,
	0x03, 0x46, 0x6C, 0x61, 0xCF, 0xC4, 0xBD, 0x0D, 0x1A, 0x9F, 0xB4, 0xAA, 0x03, 0x84, 0x8D, 0x15,
	0x3E, 0x8F, 0xBA, 0x28, 0x94, 0x09, 0x35, 0x28, 0xE5, 0x15, 0xBC, 0xAF, 0x33, 0xBA, 0x67, 0xF2,
	0x06, 0x79, 0xEE, 0x50, 0x0F, 0x14, 0x98, 0xFC, 0x95, 0xEC, 0x65, 0x40, 0x88, 0xA8, 0x1A, 0x0C,
	0x10, 0x74, 0x79, 0x42, 0x3B, 0xCD, 0xE1, 0xD1, 0xAD, 0x7E, 0x29, 0x41, 0xC4, 0x39, 0x75, 0xC5,
	0xCB, 0x0F, 0xB1, 0x6F, 0x30, 0xD3, 0xAE, 0x53, 0x59, 0xD6, 0x86, 0x34, 0x31, 0x8B, 0x96, 0x82,
	0xDF, 0xA4, 0x01, 0x32, 0xB4, 0x29, 0xDC, 0x9C, 0x28, 0x53, 0x72, 0xAE, 0x96, 0x37, 0xE3, 0x65,
	0x59, 0x91, 0x84, 0x95, 0xB3, 0x2D, 0x3F, 0x84, 0x12, 0xD2, 0x52, 0x85, 0x8D, 0x85, 0xD5, 0x2E,
	0x2A, 0x3E, 0xEB, 0x0C, 0x11, 0xA4, 0x4F, 0xED, 0x29, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x69,
	0x30, 0x67, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x1D, 0x13, 0x01, 0x01, 0xFF, 0x04, 0x05, 0x30, 0x03,
	0x01, 0x01, 0xFF, 0x30, 0x54, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x4D, 0x30, 0x4B, 0x80, 0x10,
	0x8F, 0x11, 0x13, 0x21, 0xAA, 0xC0, 0xFA, 0xB1, 0x63, 0xD5, 0xE6, 0x00, 0x9B, 0x78, 0x67, 0x40,
	0xA1, 0x25, 0x30, 0x23, 0x31, 0x21, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x1E, 0x18, 0x00,
	0x44, 0x00, 0x43, 0x00, 0x53, 0x00, 0x5F, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x61, 0x00, 0x74, 0x00,
	0x66, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x6D, 0x82, 0x10, 0x32, 0xDC, 0x46, 0x30, 0x87, 0xE5, 0x4F,
	0xB1, 0x43, 0x0F, 0x58, 0x9E, 0xC0, 0xDA, 0x58, 0xF8, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48,
	0x86, 0xF7, 0x0D, 0x01, 0x01, 0x0B, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0x7D, 0x03, 0x2A,
	0x49, 0x7E, 0x0C, 0x43, 0x4E, 0xAE, 0x45, 0xDD, 0xE2, 0x62, 0xB2, 0x31, 0x55, 0xEB, 0x6C, 0xF8,
	0x96, 0xFC, 0x5A, 0x5F, 0xA7, 0xD2, 0x26, 0xA5, 0x10, 0x15, 0x85, 0x1D, 0xDE, 0xCD, 0x97, 0xFB,
	0x6D, 0x19, 0xED, 0x03, 0x93, 0x83, 0x94, 0x04, 0x1B, 0xE6, 0x00, 0xBA, 0x41, 0xCF, 0xAB, 0xB7,
	0x46, 0x17, 0x3F, 0x8E, 0x3B, 0x2D, 0xC4, 0x54, 0x67, 0x31, 0x11, 0x0D, 0xA4, 0x35, 0x1E, 0xC4,
	0x09, 0xC2, 0xCB, 0xFD, 0x19, 0x1B, 0x5B, 0x2A, 0x19, 0x6A, 0xB9, 0x72, 0x3E, 0x27, 0x8A, 0x0A,
	0xBD, 0xB4, 0x68, 0x5D, 0xA9, 0x72, 0xC7, 0x0E, 0x28, 0x06, 0xC9, 0x4C, 0xE1, 0x56, 0xEB, 0x15,
	0x16, 0xC1, 0xD2, 0x86, 0x63, 0x57, 0xB1, 0xAA, 0x01, 0xF9, 0x26, 0xBC, 0xA7, 0xED, 0x0D, 0x02,
	0x80, 0xA7, 0x77, 0x57, 0xE9, 0xA5, 0x3B, 0x72, 0xC2, 0xAA, 0x6D, 0x7B, 0xA8, 0x40, 0xA3, 0x34,
	0x7B, 0x73, 0x40, 0x90, 0xFC, 0x43, 0x00, 0x29, 0x97, 0x7C, 0x41, 0xB2, 0xCA, 0x31, 0xA7, 0x86,
	0x08, 0xDF, 0x67, 0xCA, 0x1B, 0xEC, 0x0C, 0x53, 0xD4, 0x0B, 0x4A, 0x22, 0x40, 0x44, 0xA8, 0xE9,
	0x9D, 0x49, 0x01, 0xC6, 0x77, 0x15, 0x6E, 0x8A, 0x1F, 0xFF, 0x42, 0xF3, 0xDE, 0xF7, 0x93, 0xFA,
	0x81, 0x8F, 0x98, 0x6B, 0x75, 0x27, 0xA8, 0xBE, 0xE9, 0x2C, 0x70, 0x0F, 0xE6, 0xA5, 0xDD, 0x5D,
	0xA5, 0x33, 0x54, 0xEE, 0xFE, 0x6F, 0x91, 0xE8, 0xB4, 0x1A, 0x55, 0x77, 0xA1, 0x98, 0x56, 0x48,
	0x9C, 0xF2, 0xA3, 0x96, 0xD7, 0xB2, 0x86, 0x15, 0xA9, 0xCA, 0xBD, 0x04, 0x1B, 0x14, 0x11, 0xBE,
	0x5D, 0xC5, 0x2C, 0x5E, 0x5B, 0x57, 0x87, 0x9B, 0xCA, 0xE8, 0xA1, 0x7F, 0x6D, 0xED, 0x79, 0x2D,
	0x89, 0x3E, 0x70, 0x3C, 0x9E, 0x5C, 0x0F, 0x26, 0xCD, 0x2D, 0xE3, 0x47, 0x6E, 0x89, 0x05, 0x5C,
	0x73, 0x03, 0x87, 0x8C, 0x44, 0xE5, 0xC5, 0x6C, 0x09, 0x8B, 0x93, 0xBC, 0x1E, 0x0F, 0x56, 0x80,
	0x45, 0xDD, 0xDA, 0x96, 0x01, 0x48, 0x7C, 0xD2, 0xC0, 0x86, 0xD1, 0x8D, 0x7C, 0xBF, 0x48, 0x74,
	0x97, 0x8F, 0x4A, 0xBE, 0xC2, 0x71, 0x29, 0x91, 0xCF, 0x6A, 0x39, 0xBE, 0xD8, 0x50, 0x75, 0xCF,
	0x24, 0x8D, 0x5A, 0x12, 0x16, 0xA8, 0x5C, 0x6C, 0x88, 0x3E, 0x9F, 0x38, 0xDE, 0x04, 0x7F, 0x89,
	0xE7, 0x5A, 0x36, 0x6D, 0xAB, 0xF3, 0xC8, 0x32, 0x64, 0x91, 0x95, 0x12, 0x69, 0x7E, 0x71, 0x09,
	0xD1, 0xDA, 0xC9, 0x5E, 0xFC, 0xF4, 0x6C, 0x38, 0x71, 0x21, 0x62, 0x50, 0xC8, 0x14, 0x47, 0x25,
	0x94, 0x67, 0xD2, 0x20, 0x45, 0xC3, 0x50, 0x43, 0x81, 0x1D, 0x56, 0xAC, 0x2A, 0x02, 0x6E, 0x6D,
	0x06, 0xCA, 0x42, 0xC9, 0x65, 0x4C, 0xF7, 0x94, 0xF7, 0x67, 0x9C, 0x24, 0x98, 0x20, 0x55, 0x6A,
	0x0D, 0x85, 0x47, 0x2F, 0x3D, 0xFC, 0xA1, 0x28, 0xFE, 0xDF, 0x6F, 0xB1, 0x31, 0x62, 0x22, 0x8F,
	0x74, 0x3E, 0x1C, 0xE0, 0x02, 0xEF, 0xF9, 0x6B, 0x10, 0x32, 0xC5, 0xF5, 0x08, 0x51, 0xC7, 0x23,
	0xE7, 0x53, 0xEA, 0x89, 0x3A, 0xB2, 0xD9, 0x8A, 0x5E, 0xB0, 0x35, 0x06, 0x0A, 0x4F, 0xEE, 0x48,
	0x79, 0x7A, 0xEE, 0xEE, 0xAF, 0x9D, 0xF6, 0x59, 0xD6, 0x25, 0x86, 0xAC, 0x05, 0x9D, 0xA7, 0x61,
	0x31, 0xE3, 0xC1, 0xD0, 0x78, 0x9F, 0x83, 0x1F, 0x7C, 0x17, 0x50, 0x05, 0xAD, 0x40, 0x1A, 0x0C,
	0x19, 0x9E, 0xE1, 0x5D, 0x83, 0xE2, 0xAB, 0x83, 0x17, 0x84, 0x13, 0x76, 0x4F, 0x29, 0xBC, 0xA6,
	0x3F, 0xAE, 0x0D, 0xF9, 0x79, 0x11, 0xF8, 0x04, 0x79, 0x94, 0x88, 0x3F, 0x0D, 0x6C, 0x1F, 0x07,
	0x61, 0xF6, 0x51, 0xB2, 0xBC, 0xB8, 0xD3, 0x87, 0xA7, 0x15, 0x12, 0x60, 0x7B
};